package easik.sketch.util.Export.Database;


import java.sql.*;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.ListIterator;

import javax.swing.JOptionPane;

import com.mysql.jdbc.CommunicationsException;

import easik.sketch.util.Export.ExportConstants;
import easik.sketch.util.Export.ExportHandler;
import easik.sketch.util.Export.SketchToSchema;

/**
 * A class to convert a sketch to a database through an existing database platform. 
 * 
 * @author Vera Ranieri 2006
 * @since 2006-05-10 Vera Ranieri
 * @version 2006-06-15 Vera Ranieri
 */
public class DatabaseBuilder {
	/**
	 * The connection made to the database
	 */
	private Connection _con;
	/**
	 * The name of the database
	 */
	private String _name;
	/**
	 * The database used
	 */
	private String _dataTypePlatform;
	/**
	 * The schema built from the XML file
	 */
	private SketchToSchema _schema;
	/**
	 * The handler made from the file
	 */
	private ExportHandler _sqlHandler;
	/**
	 * The UI used to gather information to form the connection
	 */
	private DatabaseUI _db;
	
	/**
	 * Creates a new database builder which forms a connection to the database.  It then adds all relevant information 
	 * to the database.
	 * @param sh The export handler for this sketch
	 * @param db The UI containing information about the connection
	 */
	public DatabaseBuilder(ExportHandler sh, DatabaseUI db){
		
		_sqlHandler = sh;
		_db = db;
		_dataTypePlatform = sh.getPlatform();
		
		try{
			Class.forName((String)ExportConstants.DRIVER_LOCATIONS.get(_dataTypePlatform));
		}
		catch(Exception e){
			e.printStackTrace();
		}
		
		try{
			String URL = checkURL(db.getURL());
			_con = DriverManager.getConnection(ExportConstants.SQL_PLATFORM_LOCATIONS.get(_dataTypePlatform) 
			                                                                      + URL, db.getUsername(), db.getPassword());
			if(setDatabase()){
				
				if(setTables()){
					if(setConstraints()){
						if(setTriggers()){
							JOptionPane.showMessageDialog(null, "Database Successfully Created!", "Success", 
									JOptionPane.INFORMATION_MESSAGE);
						}
					}
				}
			}
		}
		catch(SQLException s){
			if(s instanceof CommunicationsException){
				JOptionPane.showMessageDialog(null,"Unknown Host.  Please ensure the URL is correct.", "Error",  JOptionPane.ERROR_MESSAGE);
			}
			
			else{
				JOptionPane.showMessageDialog(null,"Invalid Username or Password", "Error",  JOptionPane.ERROR_MESSAGE);
				s.printStackTrace();
			}
		}

		catch(Exception e){
			JOptionPane.showMessageDialog(null, "Error creating database.  Database not created.", "Error", JOptionPane.ERROR_MESSAGE);
			e.printStackTrace();
			
		}
		
	}
	
	/**
	 * Class to prompt the user for the URL of the database.  Returns null if it is an invalid address
	 * @param url The URL provided by the user
	 * @return String URL formatted for database connection
	 * @throws Exception If the URL is null or contains spaces
	 */
	private String checkURL(String url) throws Exception{
		
		//TODO: More Error checking.
		if(url.equals(null) || url.indexOf(" ")!=-1){
			throw new Exception();
		}
		
		if(!url.startsWith("//"))
			url = "//" + url;
		if(!url.endsWith("/"))
			url += "/";
		
		return url;
	}
	
	/**
	 * Method to create a new database and set this database to be the one to use 
	 * from all points afterwords
	 * 
	 * @return True if the database was successfully set, false otherwise
	 */
	private boolean setDatabase(){
		
		_name = _db.getDb();
		_name = _name.trim();
		if(_name.indexOf(" ") != -1){
			_name = _name.replace(' ', '_');
			JOptionPane.showMessageDialog(null, "Spaces in database name are not allowed.\n" 
					+"Database will be named "+ _name, "Warning", JOptionPane.WARNING_MESSAGE);
		}
		
		_schema = new SketchToSchema(_sqlHandler, _name);
		
		try {
			String command = _schema.getCreateString();
			
			Statement s = _con.createStatement();
			s.executeUpdate(command);
			
			command = _schema.getUseString();
			s.executeUpdate(command);
			
			return true;
		}
		catch (SQLException e) {
			JOptionPane.showMessageDialog(null, "Error creating database " + _name +"\n"+
					"Please ensure that database does not already exist.\n" +
					"Full database not created.",
					"Error", JOptionPane.ERROR_MESSAGE);
			e.printStackTrace();
			return false;
		}
		
		
	}
	
	/**
	 * A method to create the database tables.
	 * 
	 * @return true if the tables were successfully set, false otherwise
	 */
	private boolean setTables(){
		
		LinkedList tables = _schema.getTables();
		
		String str = "";
		
		ListIterator li = tables.listIterator();

		try{
			Statement s = _con.createStatement();
			while(li.hasNext()){
				
				str = (String) li.next();
				s.executeUpdate(str);
				
			}
		}
		
		catch (SQLException e) {
			JOptionPane.showMessageDialog(null, "Invalid SQL syntax with statement: \n"+
					str + "\n Full Database Not Created","Error", JOptionPane.ERROR_MESSAGE);
			e.printStackTrace();
			return false;
		}
		
		return(setFailTable());
		
	}
	
	/**
	 * Sets the fail table and procedure for the newly created database so failures can be logged and triggers can produce an 
	 * error statement should an illegal operation be performed.
	 * 
	 * @return True if the fail table was successfully created, false otherwise
	 * @since 2006-06-15 Vera Ranieri
	 */
	private boolean setFailTable(){
		
		try{
			Statement s = _con.createStatement();
			s.executeUpdate(ExportConstants.FAIL_TABLE);
			s.executeUpdate(ExportConstants.FAIL_PROC);
			return true;
		}
		catch(SQLException s){
			s.printStackTrace();
			return false;
		}
		
	}
	
	/**
	 * Sets the constraint string in the database.
	 * 
	 * @return True if constraints were successfully set, false otherwise
	 * @since 2006-06-14 Vera Ranieri
	 */
	private boolean setConstraints(){
		
		ArrayList constraints = _schema.getConstraints();
		String str;
		Statement s;
		for(int i =0; i< constraints.size(); i++){
			
			str = (String) constraints.get(i);
			try {
				s = _con.createStatement();
				s.executeUpdate(str);
			} catch (SQLException e) {
				JOptionPane.showMessageDialog(null, "Invalid SQL syntax with statement: \n"+
						str + "\n Full Database Not Created","Error", JOptionPane.ERROR_MESSAGE);
				e.printStackTrace();
				return false;
			}
			
		}
		
		return true;
		
	}
	
	/**
	 * A method to pass trigger strings to the SQL server.
	 * 
	 * @return True if all triggers were successfully accepted by the server, false otherwise
	 * @since 2006-06-15 Vera Ranieri
	 */
	private boolean setTriggers(){
		ArrayList triggers = _schema.getTriggers();
		String str;
		Statement s;
		for(int i =0; i< triggers.size(); i++){
			
			str = (String) triggers.get(i);
			try {
				s = _con.createStatement();
				s.executeUpdate(str);
			} catch (SQLException e) {
				JOptionPane.showMessageDialog(null, "Invalid SQL syntax with statement: \n"+
						str + "\n Full Database Not Created","Error", JOptionPane.ERROR_MESSAGE);
				e.printStackTrace();
				return false;
			}
			
		}
		
		return true;
	}
}

